<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\MenuManager;
use App\Services\PluginManager;

class CleanMenuCommand extends Command
{
    protected $signature = 'menu:clean {--force : Force cleanup without confirmation}';
    protected $description = 'Clean orphaned menu items from inactive plugins';

    public function handle()
    {
        $this->info('🧹 Cleaning menu items...');
        
        $menuManager = app(MenuManager::class);
        $pluginManager = app(PluginManager::class);
        
        $allMenuItems = $menuManager->getMenuItems();
        $activePlugins = $pluginManager->getActivePlugins();
        
        $this->info("📊 Current state:");
        $this->info("   Active plugins: " . count($activePlugins));
        $totalItems = array_sum(array_map('count', $allMenuItems));
        $this->info("   Total menu items: " . $totalItems);
        
        $orphanedItems = [];
        foreach ($allMenuItems as $category => $items) {
            foreach ($items as $item) {
                if (isset($item['plugin']) && !in_array($item['plugin'], $activePlugins)) {
                    $orphanedItems[] = [
                        'category' => $category,
                        'plugin' => $item['plugin'],
                        'label' => $item['label']
                    ];
                }
            }
        }
        
        if (empty($orphanedItems)) {
            $this->info("✅ No orphaned menu items found. Everything is clean!");
            return;
        }
        
        $this->warn("🔍 Found " . count($orphanedItems) . " orphaned menu items:");
        foreach ($orphanedItems as $item) {
            $this->line("   - {$item['label']} (Plugin: {$item['plugin']}, Category: {$item['category']})");
        }
        
        if (!$this->option('force') && !$this->confirm('Do you want to remove these orphaned items?')) {
            $this->info('Cleanup cancelled.');
            return;
        }
        
        $cleaned = $menuManager->cleanupOrphanedItems();
        
        if ($cleaned) {
            $this->info("✅ Cleanup completed successfully!");
            
            $newMenuItems = $menuManager->getMenuItems();
            $newTotalItems = array_sum(array_map('count', $newMenuItems));
            $this->info("📊 After cleanup:");
            $this->info("   Total menu items: " . $newTotalItems);
            $this->info("   Removed: " . ($totalItems - $newTotalItems) . " items");
        } else {
            $this->warn("⚠️  Cleanup completed but no changes were made.");
        }
        
        if ($this->option('force') || $this->confirm('Do you want to re-register active plugin menus?', false)) {
            $this->info("🔄 Re-registering active plugin menus...");
            
            foreach ($activePlugins as $pluginName) {
                try {
                    $pluginManager->registerPluginAdminMenu($pluginName);
                    $this->line("   ✓ Re-registered: {$pluginName}");
                } catch (\Exception $e) {
                    $this->error("   ✗ Failed to re-register: {$pluginName} - " . $e->getMessage());
                }
            }
        }
        
        $this->info("🎉 Menu cleanup completed!");
    }
}